using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class represents DivisionInfoType group described 
	/// in VistALink protocol specification. Used to represent minimal division information. 
	/// </summary>
	public class DivisionInfo
	{
		private string _ien;
		private string _name;
		private string _number;

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_SINGLE_DIVISION_NODE_NAME = "Division";
		private const string XMLCONSTS_IEN_DIV_ATTRIBUTE_NAME = "ien";
		private const string XMLCONSTS_NAME_DIV_ATTRIBUTE_NAME = "divName";
		private const string XMLCONSTS_NUMBER_DIV_ATTRIBUTE_NAME = "divNumber";
		private const string XMLCONSTS_DEFAULT_DIV_ATTRIBUTE_NAME = "default";
		private const string XMLCONSTS_TRUE = "true";

		/// <summary>
		/// Constructor defining all parameters required to create a division info object. 
		/// </summary>
		/// <param name="ien">Internal entry number of division.</param>
		/// <param name="name">Division name.</param>
		/// <param name="number">Division number.</param>
		public DivisionInfo( string ien, string name, string number )
		{
			if( ien == null ) 
				throw( new ArgumentNullException( "ien" ) );

			if( ien.Trim() == String.Empty )
				throw( new ArgumentOutOfRangeException( "ien", SR.Exceptions.ParameterCannotBeBlankOrSpacesOnly( "Internal entry number of division " ) ) );

			if( name == null ) 
				throw( new ArgumentNullException( "name" ) );

			if( name.Trim() == String.Empty )
				throw( new ArgumentOutOfRangeException( "name", SR.Exceptions.ParameterCannotBeBlankOrSpacesOnly( "Division name " ) ) );

			if( number == null ) 
				throw( new ArgumentNullException( "number" ) );

			if( number.Trim() == String.Empty ) 
				throw( new ArgumentOutOfRangeException( "number", SR.Exceptions.ParameterCannotBeBlankOrSpacesOnly( "Division number " ) ) );

			_ien = ien;
			_name = name;
			_number = number;
		}

		// TODO: Add unit tests. 
		/// <summary>
		/// XML deserialization factory method retrieving a division information XML element 
		/// from a list of descendants of a given node by its unique tag name, parsing it
		/// and returning division info object created from it.
		/// </summary>
		/// <param name="rootElement">XML node containing division info element among its children.</param>
		/// <returns>Remote user info object deserialized from XML.</returns>
		public static DivisionInfo ParseFindChildNodeByUniqueTagName( XmlElement rootElement )
		{
			if( rootElement == null )
				throw( new ArgumentNullException( "rootElement" ) );

			return Parse( XmlUtility.ParseGetRequiredElementByUniqueTagName( rootElement, XMLCONSTS_SINGLE_DIVISION_NODE_NAME ) );
		}

		/// <summary>
		/// XML deserialization factory method creating an instance of class from supplied XML node. 
		/// </summary>
		/// <param name="divisionNode">XmlNode to use.</param>
		/// <returns>Instance of class deserialized from a given XmlNode.</returns>
		public static DivisionInfo Parse( XmlNode divisionNode )
		{
			bool _markedAsDefaultStub;
			return DivisionInfo.Parse( divisionNode, out _markedAsDefaultStub );
		}

		/// <summary>
		/// XML deserialization factory method creating an instance of class from supplied XML node. 
		/// </summary>
		/// <param name="divisionNode">XmlNode to use.</param>
		/// <param name="markedAsDefault">Out parameter used to indicate whether division node was marked as "default"</param>
		/// <returns>Instance of class deserialized from a given XmlNode.</returns>
		public static DivisionInfo Parse( XmlNode divisionNode, out bool markedAsDefault )
		{
			if( divisionNode == null )
				throw( new ArgumentNullException( "divisionNode" ) );

			XmlElement _divisionElement = XmlUtility.ParseValidateConvertNodeToElement( divisionNode, XMLCONSTS_SINGLE_DIVISION_NODE_NAME );

			markedAsDefault = _divisionElement.HasAttribute( XMLCONSTS_DEFAULT_DIV_ATTRIBUTE_NAME );

			if( markedAsDefault )
				XmlUtility.ParseCheckRequiredAttributeValue( _divisionElement, XMLCONSTS_DEFAULT_DIV_ATTRIBUTE_NAME, XMLCONSTS_TRUE );

			try
			{
				return new DivisionInfo(
					XmlUtility.ParseGetRequiredAttributeValue( _divisionElement, XMLCONSTS_IEN_DIV_ATTRIBUTE_NAME ),
					XmlUtility.ParseGetRequiredAttributeValue( _divisionElement, XMLCONSTS_NAME_DIV_ATTRIBUTE_NAME ),
					XmlUtility.ParseGetRequiredAttributeValue( _divisionElement, XMLCONSTS_NUMBER_DIV_ATTRIBUTE_NAME ) );
			}
			catch( ArgumentException xcp )
			{
				throw( new XmlParseException( 
					SR.Exceptions.XmlDeserializationFailedDueToInnerException( typeof(DivisionInfo).Name, _divisionElement.Name ), xcp ) );
			}
		}
		/// <summary>
		/// XML serialization method writing XML representation of division info 
		/// into supplied XmlWriter.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		public void WriteDivisionInfoToXml( XmlWriter writer )		
		{
			this.WriteDivisionInfoToXml( writer, false );
		}

		/// <summary>
		/// XML serialization method writing XML representation of division info 
		/// into supplied XmlWriter.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		/// <param name="markDefault">Flag indicated whether division info node should be marked with "default" attribute.</param>
		public void WriteDivisionInfoToXml( XmlWriter writer, bool markDefault )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_SINGLE_DIVISION_NODE_NAME );

			writer.WriteAttributeString( XMLCONSTS_IEN_DIV_ATTRIBUTE_NAME, _ien );
			writer.WriteAttributeString( XMLCONSTS_NAME_DIV_ATTRIBUTE_NAME, _name );
			writer.WriteAttributeString( XMLCONSTS_NUMBER_DIV_ATTRIBUTE_NAME, _number );

			if( markDefault )
				writer.WriteAttributeString( XMLCONSTS_DEFAULT_DIV_ATTRIBUTE_NAME, XMLCONSTS_TRUE );

			writer.WriteEndElement();
		}

		/// <summary>
		/// Internal entry number of division.
		/// </summary>
		public string Ien
		{
			get
			{
				return _ien;
			}
		}

		/// <summary>
		/// Division name.
		/// </summary>
		public string Name
		{
			get
			{
				return _name;
			}
		}

		/// <summary>
		/// Division number.
		/// </summary>
		public string Number 
		{
			get
			{
				return _number;
			}
		}
	}
}
